#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifndef __GNUC__
#include <io.h>
#else
#include <unistd.h>
#include <unixstr.h>
#endif


#include <swmgr.h>
#include <rawtext.h>
#include <rawgbf.h>
#include <rawcom.h>
#include <hrefcom.h>
#include <rawld.h>
#include <utilfuns.h>
#include <gbfrtf.h>
#include <rwprtf.h>
#include <rawfiles.h>


SWMgr::SWMgr(SWConfig *iconfig, bool autoload) {
	gbftortf = new GBFRTF();
	rwptortf = new RWPRTF();
	
	if (iconfig) {
		config   = iconfig;
		myconfig = 0;
	}
	else config = myconfig = new SWConfig("./mods.conf");
	if (autoload)
		Load();
}


void SWMgr::Load() {
	SectionMap::iterator Sectloop, Sectend;
	ConfigEntMap::iterator Entryloop, Entryend;

	DeleteMods();

	for (Sectloop = myconfig->Sections.lower_bound("Globals"), Sectend = myconfig->Sections.upper_bound("Globals"); Sectloop != Sectend; Sectloop++) {		// scan thru all 'Globals' sections
		for (Entryloop = (*Sectloop).second.lower_bound("AutoInstall"), Entryend = (*Sectloop).second.upper_bound("AutoInstall"); Entryloop != Entryend; Entryloop++)	// scan thru all AutoInstall entries
			InstallScan((*Entryloop).second.c_str());		// Scan AutoInstall entry directory for new modules and install
	}		
	config->Load();		// force reload on config object because we may have installed new modules
	CreateMods();
}

SWMgr::~SWMgr() {

	DeleteMods();

	if (gbftortf)
		delete gbftortf;
			
	if (rwptortf)
		delete rwptortf;
			
	if (myconfig)
		delete myconfig;
}


void SWMgr::CreateMods() {
	SectionMap::iterator it;
	SWModule *newmod;
	string description, driver, datapath, misc1;
	for (it = config->Sections.begin(); it != config->Sections.end(); it++) {
		newmod = 0;
		
		description = (*it).second["Description"];
		driver = (*it).second["ModDrv"];
		datapath = (*it).second["DataPath"];
		if (!driver.empty()) {
			if (!stricmp(driver.c_str(), "RawText")) {
				newmod = new RawText(datapath.c_str(), (*it).first.c_str(), description.c_str());
			}
			
			if (!stricmp(driver.c_str(), "RawGBF")) {
				newmod = new RawGBF(datapath.c_str(), (*it).first.c_str(), description.c_str());
				newmod->AddRenderFilter(gbftortf);
			}

			if (!stricmp(driver.c_str(), "RawCom")) {
				newmod = new RawCom(datapath.c_str(), (*it).first.c_str(), description.c_str());
			}
						
			if (!stricmp(driver.c_str(), "RawFiles")) {
				newmod = new RawFiles(datapath.c_str(), (*it).first.c_str(), description.c_str());
			}
						
			if (!stricmp(driver.c_str(), "HREFCom")) {
				misc1 = (*it).second["Prefix"];
				newmod = new HREFCom(datapath.c_str(), misc1.c_str(), (*it).first.c_str(), description.c_str());
			}
						
			if (!stricmp(driver.c_str(), "RawLD"))
				newmod = new RawLD(datapath.c_str(), (*it).first.c_str(), description.c_str());

			if (newmod) {
				if (!stricmp((*it).first.c_str(), "RWP"))
					newmod->AddRenderFilter(rwptortf);
				Modules.insert(ModMap::value_type(newmod->Name(), newmod));
			}
		}
	}
}


void SWMgr::DeleteMods() {

	ModMap::iterator it;

	for (it = Modules.begin(); it != Modules.end(); it++)
		delete (*it).second;

	Modules.erase(Modules.begin(), Modules.end());
}


void SWMgr::InstallScan(const char *dirname)
{
   DIR *dir;
   struct dirent *ent;
   int conffd = 0;
   string newmodfile;
 
	if ((dir = opendir(dirname))) {
		rewinddir(dir);
		while ((ent = readdir(dir))) {
			if ((strcmp(ent->d_name, ".")) && (strcmp(ent->d_name, ".."))) {
				if (conffd < 1) {
					conffd = open(config->filename.c_str(), O_WRONLY|O_APPEND);
					if (conffd > 0)
						lseek(conffd, 0L, SEEK_END);
				}
				newmodfile = dirname;
				newmodfile += ent->d_name;
				AddModToConfig(conffd, newmodfile.c_str());
				unlink(newmodfile.c_str());
			}
		}
		if (conffd > 0)
			close(conffd);
		closedir(dir);
	}
}


char SWMgr::AddModToConfig(int conffd, const char *fname)
{
	int modfd;
	char ch;

	SWLog::systemlog->LogTimedInformation("Found new module [%s]. Installing...", fname);
	modfd = open(fname, O_RDONLY);
	ch = '\n';
	write(conffd, &ch, 1);
	while (read(modfd, &ch, 1) == 1)
		write(conffd, &ch, 1);
	ch = '\n';
	write(conffd, &ch, 1);
	close(modfd);
	return 0;
}
